// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CRYPTO_SYMMETRIC_KEY_H_
#define CRYPTO_SYMMETRIC_KEY_H_

#include <stddef.h>

#include <memory>
#include <string>

#include "base/containers/span.h"
#include "build/build_config.h"
#include "crypto/crypto_export.h"

namespace crypto {

// A SymmetricKey is an array of bytes which is used for symmetric cryptography
// (encryption or MACs).
//
// This whole type is deprecated: prefer to use raw std::array<uint8_t>,
// std::vector<uint8_t>, or base::span<uint8_t> instead. This type has no
// behavior or particular meaning.
//
// TODO(https://issues.chromium.org/issues/370724578): get rid of this.
class CRYPTO_EXPORT SymmetricKey {
 public:
  // Defines the algorithm that a key will be used with. See also
  // class Encryptor.
  enum Algorithm {
    AES,
    HMAC_SHA1,
  };

  SymmetricKey() = delete;

  // Wrap the given span of bytes as a SymmetricKey.
  explicit SymmetricKey(base::span<const uint8_t> key_bytes);
  virtual ~SymmetricKey();

  SymmetricKey(const SymmetricKey&);
  SymmetricKey& operator=(const SymmetricKey&);

  // Generates a random key suitable to be used with |algorithm| and of
  // |key_size_in_bits| bits. |key_size_in_bits| must be a multiple of 8.
  //
  // Deprecated: use the value version below that does not take an algorithm.
  static std::unique_ptr<SymmetricKey> GenerateRandomKey(
      Algorithm algorithm,
      size_t key_size_in_bits);

  static SymmetricKey RandomKey(size_t key_size_in_bits);

  // Derives a key from the supplied password and salt using PBKDF2, suitable
  // for use with specified |algorithm|. Note |algorithm| is not the algorithm
  // used to derive the key from the password. |key_size_in_bits| must be a
  // multiple of 8. The caller is responsible for deleting the returned
  // SymmetricKey.
  //
  // Deprecated: use crypto::kdf::DeriveKeyPbkdf2HmacSha1() instead.
  static std::unique_ptr<SymmetricKey> DeriveKeyFromPasswordUsingPbkdf2(
      Algorithm algorithm,
      const std::string& password,
      const std::string& salt,
      size_t iterations,
      size_t key_size_in_bits);

  // Derives a key from the supplied password and salt using scrypt, suitable
  // for use with specified |algorithm|. Note |algorithm| is not the algorithm
  // used to derive the key from the password. |cost_parameter|, |block_size|,
  // and |parallelization_parameter| correspond to the parameters |N|, |r|, and
  // |p| from the scrypt specification (see RFC 7914). |key_size_in_bits| must
  // be a multiple of 8. The caller is responsible for deleting the returned
  // SymmetricKey.
  //
  // Deprecated: use crypto::kdf::DeriveKeyScrypt() instead.
  // Warning: this function will CHECK() that the passed in parameters are
  // valid, and the definition of 'valid' is subtle. Be careful using it.
  static std::unique_ptr<SymmetricKey> DeriveKeyFromPasswordUsingScrypt(
      Algorithm algorithm,
      const std::string& password,
      const std::string& salt,
      size_t cost_parameter,
      size_t block_size,
      size_t parallelization_parameter,
      size_t max_memory_bytes,
      size_t key_size_in_bits);

  // Imports an array of key bytes in |raw_key|. This key may have been
  // generated by GenerateRandomKey or DeriveKeyFromPassword{Pbkdf2,Scrypt} and
  // exported with key(). The key must be of suitable size for use with
  // |algorithm|. The caller owns the returned SymmetricKey.
  //
  // Deprecated: use the regular constructor that accepts a span of bytes, and
  // validate that the key is of whatever length your client code expects before
  // doing so.
  static std::unique_ptr<SymmetricKey> Import(Algorithm algorithm,
                                              const std::string& raw_key);

  // Returns the raw platform specific key data.
  const std::string& key() const { return key_; }

 private:
  std::string key_;
};

}  // namespace crypto

#endif  // CRYPTO_SYMMETRIC_KEY_H_
