/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <QJsonObject>

#include "src/datovka_shared/json/basic.h"
#include "src/datovka_shared/json/helper.h"
#include "src/json/db_info.h"

#define nullInt 0

/*!
 * @brief PIMPL Json::DbInfo class.
 */
class Json::DbInfoPrivate {
public:
	DbInfoPrivate(void)
	    : m_formatVersionMajor(nullInt), m_formatVersionMinor(nullInt)
	{ }

	DbInfoPrivate &operator=(const DbInfoPrivate &other) Q_DECL_NOTHROW
	{
		m_formatVersionMajor = other.m_formatVersionMajor;
		m_formatVersionMinor = other.m_formatVersionMinor;

		return *this;
	}

	bool operator==(const DbInfoPrivate &other) const
	{
		return (m_formatVersionMajor == other.m_formatVersionMajor) &&
		    (m_formatVersionMinor == other.m_formatVersionMinor);
	}

	int m_formatVersionMajor;
	int m_formatVersionMinor;
};

Json::DbInfo::DbInfo(void)
    : Object(),
    d_ptr(Q_NULLPTR)
{
}

Json::DbInfo::DbInfo(const DbInfo &other)
    : Object(),
    d_ptr((other.d_func() != Q_NULLPTR) ? (new (::std::nothrow) DbInfoPrivate) : Q_NULLPTR)
{
	Q_D(DbInfo);
	if (d == Q_NULLPTR) {
		return;
	}

	*d = *other.d_func();
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::DbInfo::DbInfo(DbInfo &&other) Q_DECL_NOEXCEPT
    : Object(),
#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
    d_ptr(other.d_ptr.release()) //d_ptr(::std::move(other.d_ptr))
#else /* < Qt-5.12 */
    d_ptr(other.d_ptr.take())
#endif /* >= Qt-5.12 */
{
}
#endif /* Q_COMPILER_RVALUE_REFS */

Json::DbInfo::~DbInfo(void)
{
}

/*!
 * @brief Ensures DbInfoPrivate presence.
 *
 * @note Returns if DbInfoPrivate could not be allocated.
 */
#define ensureDbInfoPrivate(_x_) \
	do { \
		if (Q_UNLIKELY(d_ptr == Q_NULLPTR)) { \
			DbInfoPrivate *p = new (::std::nothrow) DbInfoPrivate; \
			if (Q_UNLIKELY(p == Q_NULLPTR)) { \
				Q_ASSERT(0); \
				return _x_; \
			} \
			d_ptr.reset(p); \
		} \
	} while (0)

Json::DbInfo &Json::DbInfo::operator=(const DbInfo &other) Q_DECL_NOTHROW
{
	if (other.d_func() == Q_NULLPTR) {
		d_ptr.reset(Q_NULLPTR);
		return *this;
	}
	ensureDbInfoPrivate(*this);
	Q_D(DbInfo);

	*d = *other.d_func();

	return *this;
}

#ifdef Q_COMPILER_RVALUE_REFS
Json::DbInfo &Json::DbInfo::operator=(DbInfo &&other) Q_DECL_NOTHROW
{
	swap(*this, other);
	return *this;
}
#endif /* Q_COMPILER_RVALUE_REFS */

bool Json::DbInfo::operator==(const DbInfo &other) const
{
	Q_D(const DbInfo);
	if ((d == Q_NULLPTR) && ((other.d_func() == Q_NULLPTR))) {
		return true;
	} else if ((d == Q_NULLPTR) || ((other.d_func() == Q_NULLPTR))) {
		return false;
	}

	return *d == *other.d_func();
}

bool Json::DbInfo::operator!=(const DbInfo &other) const
{
	return !operator==(other);
}

bool Json::DbInfo::isNull(void) const
{
	Q_D(const DbInfo);
	return d == Q_NULLPTR;
}

bool Json::DbInfo::isValid(void) const
{
	return !isNull();
}

int Json::DbInfo::formatVersionMajor(void) const
{
	Q_D(const DbInfo);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return nullInt;
	}

	return d->m_formatVersionMajor;
}

void Json::DbInfo::setFormatVersionMajor(int maj)
{
	ensureDbInfoPrivate();
	Q_D(DbInfo);
	d->m_formatVersionMajor = maj;
}

int Json::DbInfo::formatVersionMinor(void) const
{
	Q_D(const DbInfo);
	if (Q_UNLIKELY(d == Q_NULLPTR)) {
		return nullInt;
	}

	return d->m_formatVersionMinor;
}

void Json::DbInfo::setFormatVersionMinor(int min)
{
	ensureDbInfoPrivate();
	Q_D(DbInfo);
	d->m_formatVersionMinor = min;
}

Json::DbInfo Json::DbInfo::fromJson(const QByteArray &json, bool *ok)
{
	QJsonObject jsonObj;
	if (!Helper::readRootObject(json, jsonObj)) {
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return DbInfo();
	}

	return fromJsonVal(jsonObj, ok);
}

static const QString keyFormatVersionMajor("formatVersionMajor");
static const QString keyFormatVersionMinor("formatVersionMinor");

Json::DbInfo Json::DbInfo::fromJsonVal(const QJsonValue &jsonVal, bool *ok)
{
	DbInfo tdi;

	if (Q_UNLIKELY(!jsonVal.isObject())) {
		goto fail;
	}

	{
		const QJsonObject jsonObj = jsonVal.toObject();
		{
			int valInt = 0;
			if (Q_UNLIKELY(!Helper::readInt(jsonObj,
			        keyFormatVersionMajor, valInt, Helper::ACCEPT_VALID))) {
				goto fail;
			}
			tdi.setFormatVersionMajor(valInt);
		}
		{
			int valInt = 0;
			if (Q_UNLIKELY(!Helper::readInt(jsonObj,
			        keyFormatVersionMinor, valInt, Helper::ACCEPT_VALID))) {
				goto fail;
			}
			tdi.setFormatVersionMinor(valInt);
		}
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return tdi;

fail:
	if (ok != Q_NULLPTR) {
		*ok = false;
	}
	return DbInfo();
}

bool Json::DbInfo::toJsonVal(QJsonValue &jsonVal) const
{
	QJsonObject jsonObj;

	jsonObj.insert(keyFormatVersionMajor, formatVersionMajor());
	jsonObj.insert(keyFormatVersionMinor, formatVersionMinor());

	jsonVal = jsonObj;
	return true;
}

void Json::swap(DbInfo &first, DbInfo &second) Q_DECL_NOTHROW
{
	using ::std::swap;
	swap(first.d_ptr, second.d_ptr);
}
