// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ui/accessibility/ax_relative_bounds.h"

#include "base/memory/values_equivalent.h"
#include "base/strings/string_number_conversions.h"
#include "ui/accessibility/ax_enum_util.h"
#include "ui/gfx/geometry/transform.h"

using base::NumberToString;

namespace ui {

AXRelativeBounds::AXRelativeBounds() = default;

AXRelativeBounds::~AXRelativeBounds() = default;

AXRelativeBounds::AXRelativeBounds(const AXRelativeBounds& other) {
  offset_container_id = other.offset_container_id;
  bounds = other.bounds;
  if (other.transform)
    transform = std::make_unique<gfx::Transform>(*other.transform);
}

AXRelativeBounds::AXRelativeBounds(AXRelativeBounds&& other) noexcept = default;

AXRelativeBounds& AXRelativeBounds::operator=(const AXRelativeBounds& other) {
  offset_container_id = other.offset_container_id;
  bounds = other.bounds;
  if (other.transform)
    transform = std::make_unique<gfx::Transform>(*other.transform);
  else
    transform.reset(nullptr);
  return *this;
}

bool AXRelativeBounds::operator==(const AXRelativeBounds& other) const {
  if (offset_container_id != other.offset_container_id)
    return false;
  if (bounds != other.bounds)
    return false;
  return base::ValuesEquivalent(transform, other.transform);
}

std::string AXRelativeBounds::ToString() const {
  std::string result;

  if (offset_container_id != kInvalidAXNodeID) {
    result +=
        "offset_container_id=" + NumberToString(offset_container_id) + " ";
  }

  result += "(" + NumberToString(bounds.x()) + ", " +
            NumberToString(bounds.y()) + ")-(" +
            NumberToString(bounds.width()) + ", " +
            NumberToString(bounds.height()) + ")";

  if (transform && !transform->IsIdentity())
    result += " transform=" + transform->ToString();

  return result;
}

std::ostream& operator<<(std::ostream& stream, const AXRelativeBounds& bounds) {
  return stream << bounds.ToString();
}

}  // namespace ui
