/**
    bespoke synth, a software modular synthesizer
    Copyright (C) 2021 Ryan Challinor (contact: awwbees@gmail.com)

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
**/
//
//  SynthGlobals.cpp
//  modularSynth
//
//  Created by Ryan Challinor on 11/22/12.
//
//


#include "SynthGlobals.h"
#include "ModularSynth.h"
#include "IAudioSource.h"
#include "INoteSource.h"
#include "IAudioReceiver.h"
#include "GridController.h"
#include "RollingBuffer.h"
#include "TextEntry.h"
#include "PatchCable.h"
#include "PatchCableSource.h"
#include "ChannelBuffer.h"
#include "IPulseReceiver.h"
#include "exprtk.hpp"
#include "UserPrefs.h"

#include "juce_audio_formats/juce_audio_formats.h"
#include "juce_gui_basics/juce_gui_basics.h"

#ifdef JUCE_MAC
#import <execinfo.h>
#endif

using namespace juce;

int gBufferSize = -999; //values set in SetGlobalSampleRateAndBufferSize(), setting them to bad values here to highlight any bugs
int gSampleRate = -999;
double gTwoPiOverSampleRate = -999;
double gSampleRateMs = -999;
double gInvSampleRateMs = -999;
double gBufferSizeMs = -999;
double gNyquistLimit = -999;
bool gPrintMidiInput = false;
double gTime = 1; //using a double here, so I'm going to lose nanosecond accuracy
//if I run for 4 months straight
//this means I'll lose 44100 hz sample accuracy in 7100 years of
//continuous uptime
IUIControl* gBindToUIControl = nullptr;
RetinaTrueTypeFont gFont;
RetinaTrueTypeFont gFontBold;
RetinaTrueTypeFont gFontFixedWidth;
float gModuleDrawAlpha = 255;
float gNullBuffer[kWorkBufferSize];
float gZeroBuffer[kWorkBufferSize];
float gWorkBuffer[kWorkBufferSize];
ChannelBuffer gWorkChannelBuffer(kWorkBufferSize);
IDrawableModule* gHoveredModule = nullptr;
IUIControl* gHoveredUIControl = nullptr;
IUIControl* gHotBindUIControl[10];
float gControlTactileFeedback = 0;
float gDrawScale = 1;
bool gShowDevModules = false;
float gCornerRoundness = 1;
std::array<float, (int)StepVelocityType::NumVelocityLevels> gStepVelocityLevels{};

std::random_device gRandomDevice;
bespoke::core::Xoshiro256ss gRandom(gRandomDevice);
std::uniform_real_distribution<float> gRandom01(0.0f, 1.f);
std::uniform_real_distribution<float> gRandomBipolarDist(-1.f, 1.f);

void SynthInit()
{
   std::locale::global(std::locale::classic());

   Clear(gZeroBuffer, kWorkBufferSize);

   for (int i = 0; i < 10; ++i)
      gHotBindUIControl[i] = nullptr;

   TheSynth->GetAudioFormatManager().registerBasicFormats();

   assert(kNumVoices <= 16); //assumption that we don't have more voices than midi channels

   gStepVelocityLevels[(int)StepVelocityType::Ghost] = 0.4f;
   gStepVelocityLevels[(int)StepVelocityType::Normal] = 0.8f;
   gStepVelocityLevels[(int)StepVelocityType::Accent] = 1.0f;
}

void LoadGlobalResources()
{
   gFont.LoadFont(ofToResourcePath("frabk.ttf"));
   gFontBold.LoadFont(ofToResourcePath("frabk_m.ttf"));
   gFontFixedWidth.LoadFont(ofToResourcePath("iosevka-type-light.ttf"));
   //gModuleShader.load(ofToResourcePath("shaders/module.vert"), ofToResourcePath("shaders/module.frag"));
}

void SetGlobalSampleRateAndBufferSize(int rate, int size)
{
   assert(size <= kWorkBufferSize);
   gBufferSize = size * UserPrefs.oversampling.Get();

   gSampleRate = rate * UserPrefs.oversampling.Get();
   gTwoPiOverSampleRate = TWO_PI / gSampleRate;
   gSampleRateMs = gSampleRate / 1000.0;
   gInvSampleRateMs = 1000.0 / gSampleRate;
   gBufferSizeMs = gBufferSize / gSampleRateMs;
   gNyquistLimit = gSampleRate / 2.0f;
}

std::string GetBuildInfoString()
{
   return
#if DEBUG
   "DEBUG BUILD " +
#endif
#if BESPOKE_NIGHTLY && !BESPOKE_SUPPRESS_NIGHTLY_LABEL
   "NIGHTLY " +
#endif
   juce::JUCEApplication::getInstance()->getApplicationVersion().toStdString() + " (" + std::string(__DATE__) + " " + std::string(__TIME__) + ")";
}

void DrawAudioBuffer(float width, float height, ChannelBuffer* buffer, float start, float end, float pos, float vol /*=1*/, ofColor color /*=ofColor::black*/, int wraparoundFrom /*= -1*/, int wraparoundTo /*= 0*/)
{
   ofPushMatrix();
   if (buffer != nullptr)
   {
      int numChannels = buffer->NumActiveChannels();
      for (int i = 0; i < numChannels; ++i)
      {
         DrawAudioBuffer(width, height / numChannels, buffer->GetChannel(i), start, MIN(end, buffer->BufferSize()), pos, vol, color, wraparoundFrom, wraparoundTo, buffer->BufferSize());
         ofTranslate(0, height / numChannels);
      }
   }
   ofPopMatrix();
}

void DrawAudioBuffer(float width, float height, const float* buffer, float start, float end, float pos, float vol /*=1*/, ofColor color /*=ofColor::black*/, int wraparoundFrom /*= -1*/, int wraparoundTo /*= 0*/, int bufferSize /*=-1*/)
{
   vol = MAX(.1f, vol); //make sure we at least draw something if there is waveform data

   ofPushStyle();

   ofSetLineWidth(1);
   ofFill();
   ofSetColor(255, 255, 255, 50);
   if (width > 0)
      ofRect(0, 0, width, height);
   else
      ofRect(width, 0, -width, height);

   float length = end - 1 - start;
   if (length < 0)
      length = length + wraparoundFrom - wraparoundTo;
   if (length < 0)
      length += bufferSize;

   if (length > 0)
   {
      const float kStepSize = 3;
      float samplesPerStep = length / abs(width) * kStepSize;
      start = start - (int(start) % MAX(1, int(samplesPerStep)));

      if (buffer && length > 0)
      {
         float step = width > 0 ? kStepSize : -kStepSize;
         samplesPerStep = length / width * step;

         ofSetColor(color);

         for (float i = 0; abs(i) < abs(width); i += step)
         {
            float mag = 0;
            int position = i / width * length + start;
            //rms
            int j;
            int inc = 1 + samplesPerStep / 100;
            for (j = 0; j < samplesPerStep; j += inc)
            {
               int sampleIdx = position + j;
               if (wraparoundFrom != -1 && sampleIdx > wraparoundFrom)
                  sampleIdx = sampleIdx - wraparoundFrom + wraparoundTo;
               if (bufferSize > 0)
                  sampleIdx %= bufferSize;
               mag = MAX(mag, fabsf(buffer[sampleIdx]));
            }
            mag = pow(mag, .25f);
            mag *= height / 2 * vol;
            if (mag > height / 2)
            {
               //ofSetColor(255,0,0);
               mag = height / 2;
            }
            else
            {
               //ofSetColor(color);
            }
            if (mag == 0)
               mag = .1f;
            ofLine(i, height / 2 - mag, i, height / 2 + mag);
         }

         if (pos != -1)
         {
            ofSetColor(0, 255, 0);
            int position = ofMap(pos, start, end, 0, width, true);
            ofLine(position, 0, position, height);
         }
      }
   }

   ofPopStyle();
}

void Add(float* dst, const float* src, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::add(dst, src, bufferSize);
#else
   for (int i = 0; i < bufferSize; ++i)
   {
      dst[i] += src[i];
   }
#endif
}

void Subtract(float* dst, const float* src, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::subtract(dst, src, bufferSize);
#else
   for (int i = 0; i < bufferSize; ++i)
   {
      dst[i] -= src[i];
   }
#endif
}

void Mult(float* buff, float val, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::multiply(buff, val, bufferSize);
#else
   for (int i = 0; i < bufferSize; ++i)
   {
      buff[i] *= val;
   }
#endif
}

void Mult(float* dst, const float* src, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::multiply(dst, src, bufferSize);
#else
   for (int i = 0; i < bufferSize; ++i)
   {
      dst[i] *= src[i];
   }
#endif
}

void Clear(float* buffer, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::clear(buffer, bufferSize);
#else
   bzero(buffer, bufferSize * sizeof(float));
#endif
}

void BufferCopy(float* dst, const float* src, int bufferSize)
{
#ifdef USE_VECTOR_OPS
   FloatVectorOperations::copy(dst, src, bufferSize);
#else
   memcpy(dst, src, bufferSize * sizeof(float));
#endif
}

std::string NoteName(int pitch, bool flat, bool includeOctave)
{
   int octave = pitch / 12;
   pitch %= 12;
   std::string ret = "x";
   switch (pitch)
   {
      default:
      case 0:
         ret = "C";
         break;
      case 1:
         //ret = "C#/Db";
         ret = flat ? "Db" : "C#";
         break;
      case 2:
         ret = "D";
         break;
      case 3:
         //ret = "D#/Eb";
         ret = flat ? "Eb" : "D#";
         break;
      case 4:
         ret = "E";
         break;
      case 5:
         ret = "F";
         break;
      case 6:
         //ret = "F#/Gb";
         ret = flat ? "Gb" : "F#";
         break;
      case 7:
         ret = "G";
         break;
      case 8:
         //ret = "G#/Ab";
         ret = flat ? "Ab" : "G#";
         break;
      case 9:
         ret = "A";
         break;
      case 10:
         //ret = "A#/Bb";
         ret = flat ? "Bb" : "A#";
         break;
      case 11:
         ret = "B";
         break;
   }

   if (includeOctave)
      ret += ofToString(octave - 2);

   return ret;
}

int PitchFromNoteName(std::string noteName)
{
   int octave = -2;
   if (isdigit(noteName[noteName.length() - 1]))
   {
      octave = noteName[noteName.length() - 1] - '0';
      if (noteName[noteName.length() - 2] == '-')
         octave *= -1;
      if (octave < 0)
         noteName = noteName.substr(0, noteName.length() - 2);
      else
         noteName = noteName.substr(0, noteName.length() - 1);
   }

   int pitch;
   for (pitch = 0; pitch < 12; ++pitch)
   {
      if (noteName == NoteName(pitch, false) || noteName == NoteName(pitch, true))
         break;
   }
   if (pitch == 12)
      TheSynth->LogEvent("error finding pitch for note " + noteName, kLogEventType_Error);
   return pitch + (octave + 2) * 12;
}

float Interp(float a, float start, float end)
{
   return a * (end - start) + start;
}

double GetPhaseInc(float freq)
{
   return freq * gTwoPiOverSampleRate;
}

float FloatWrap(float num, float space)
{
   if (space == 0)
      num = 0;
   num -= space * floor(num / space);
   return num;
}

double DoubleWrap(double num, float space)
{
   if (space == 0)
      num = 0;
   num -= space * floor(num / space);
   return num;
}

void DrawTextNormal(std::string text, int x, int y, float size)
{
   gFont.DrawString(text, size, x, y);
}

void DrawTextRightJustify(std::string text, int x, int y, float size)
{
   gFont.DrawString(text, size, x - gFont.GetStringWidth(text, size), y);
}

void DrawTextBold(std::string text, int x, int y, float size)
{
   gFontBold.DrawString(text, size, x, y);
}

float GetStringWidth(std::string text, float size)
{
   return gFont.GetStringWidth(text, size);
}

void AssertIfDenormal(float input)
{
   assert(input == 0 || input != input || fabsf(input) > std::numeric_limits<float>::min());
}

float GetInterpolatedSample(double offset, const float* buffer, int bufferSize)
{
   offset = DoubleWrap(offset, bufferSize);
   int pos = int(offset);
   int posNext = int(offset + 1) % bufferSize;

   float sample = buffer[pos];
   float nextSample = buffer[posNext];
   float a = offset - pos;
   float output = (1 - a) * sample + a * nextSample; //interpolate

   return output;
}

float GetInterpolatedSample(double offset, ChannelBuffer* buffer, int bufferSize, float channelBlend)
{
   assert(channelBlend <= buffer->NumActiveChannels());
   assert(channelBlend >= 0);

   if (buffer->NumActiveChannels() == 1)
      return GetInterpolatedSample(offset, buffer->GetChannel(0), bufferSize);

   int channelA = floor(channelBlend);
   if (channelA == buffer->NumActiveChannels() - 1)
      channelA -= 1;
   int channelB = channelA + 1;

   return (1 - (channelBlend - channelA)) * GetInterpolatedSample(offset, buffer->GetChannel(channelA), bufferSize) +
          (channelBlend - channelA) * GetInterpolatedSample(offset, buffer->GetChannel(channelB), bufferSize);
}

void WriteInterpolatedSample(double offset, float* buffer, int bufferSize, float sample)
{
   offset = DoubleWrap(offset, bufferSize);
   int pos = int(offset);
   int posNext = int(offset + 1) % bufferSize;

   float a = offset - pos;
   buffer[pos] += (1 - a) * sample;
   buffer[posNext] += a * sample;
}

std::string GetRomanNumeralForDegree(int degree)
{
   std::string roman;
   switch ((degree + 700) % 7)
   {
      default:
      case 0: roman = "I"; break;
      case 1: roman = "II"; break;
      case 2: roman = "III"; break;
      case 3: roman = "IV"; break;
      case 4: roman = "V"; break;
      case 5: roman = "VI"; break;
      case 6: roman = "VII"; break;
   }
   return roman;
}

void UpdateTarget(IDrawableModule* module)
{
   IAudioSource* audioSource = dynamic_cast<IAudioSource*>(module);
   INoteSource* noteSource = dynamic_cast<INoteSource*>(module);
   IGridController* grid = dynamic_cast<IGridController*>(module);
   IPulseSource* pulseSource = dynamic_cast<IPulseSource*>(module);
   std::string targetName = "";
   if (audioSource)
   {
      for (int i = 0; i < audioSource->GetNumTargets(); ++i)
      {
         IDrawableModule* target = dynamic_cast<IDrawableModule*>(audioSource->GetTarget(i));
         if (target)
            targetName = target->Path();
         else
            targetName = "";
         module->GetSaveData().SetString("target" + (i == 0 ? "" : ofToString(i + 1)), targetName);
      }
   }
   if (noteSource || grid || pulseSource)
   {
      if (module->GetPatchCableSource())
      {
         const std::vector<PatchCable*>& cables = module->GetPatchCableSource()->GetPatchCables();
         for (int i = 0; i < cables.size(); ++i)
         {
            PatchCable* cable = cables[i];
            IClickable* target = cable->GetTarget();
            if (target)
            {
               targetName += target->Path();
               if (i < cables.size() - 1)
                  targetName += ",";
            }
         }
      }
      module->GetSaveData().SetString("target", targetName);
   }
}

void DrawLissajous(RollingBuffer* buffer, float x, float y, float w, float h, float r, float g, float b, bool autocorrelationMode /* = true */)
{
   ofPushStyle();
   ofSetLineWidth(1.5f);

   int secondChannel = 1;
   if (buffer->NumChannels() == 1)
      secondChannel = 0;

   ofSetColor(r * 255, g * 255, b * 255, 70);
   ofBeginShape();
   const int delaySamps = 90;
   int numPoints = MIN(buffer->Size() - delaySamps - 1, .02f * gSampleRate);
   for (int i = 100; i < numPoints; ++i)
   {
      float vx = x + w / 2 + buffer->GetSample(i, 0) * .8f * MIN(w, h);
      float vy;
      if (autocorrelationMode)
         vy = y + h / 2 + buffer->GetSample(i + delaySamps, secondChannel) * .8f * MIN(w, h);
      else
         vy = y + h / 2 + buffer->GetSample(i, secondChannel) * .8f * MIN(w, h);
      //float alpha = 1 - (i/float(numPoints));
      //ofSetColor(r*255,g*255,b*255,alpha*alpha*255);
      ofVertex(vx, vy);
   }
   ofEndShape();

   ofPopStyle();
}

void StringCopy(char* dest, const char* source, int destLength)
{
   if (dest == source)
      return;
   strncpy(dest, source, destLength);
   dest[destLength - 1] = 0;
}

int GetKeyModifiers()
{
   int ret = 0;
   if (ModifierKeys::currentModifiers.isShiftDown())
      ret |= kModifier_Shift;
   if (ModifierKeys::currentModifiers.isAltDown())
      ret |= kModifier_Alt;
#if BESPOKE_MAC
   if (ModifierKeys::currentModifiers.isCtrlDown())
      ret |= kModifier_Control; //control and command interfere with each other on non-mac keyboards
#endif
   if (ModifierKeys::currentModifiers.isCommandDown())
      ret |= kModifier_Command;
   return ret;
}

bool IsKeyHeld(int key, int modifiers)
{
   return IKeyboardFocusListener::GetActiveKeyboardFocus() == nullptr &&
          KeyPress::isKeyCurrentlyDown(key) &&
          GetKeyModifiers() == modifiers &&
          TheSynth->GetMainComponent()->hasKeyboardFocus(true);
}

int KeyToLower(int key)
{
   if (key < CHAR_MAX && CharacterFunctions::isLetter((char)key))
      return tolower(key);
   if (key == '!')
      return '1';
   if (key == '@')
      return '2';
   if (key == '#')
      return '3';
   if (key == '$')
      return '4';
   if (key == '%')
      return '5';
   if (key == '^')
      return '6';
   if (key == '&')
      return '7';
   if (key == '*')
      return '8';
   if (key == '(')
      return '9';
   if (key == ')')
      return '0';
   if (key == '~')
      return '`';
   if (key == '_')
      return '-';
   if (key == '+')
      return '=';
   if (key == '<')
      return ',';
   if (key == '>')
      return '.';
   if (key == '?')
      return '/';
   if (key == ':')
      return ';';
   if (key == '"')
      return '\'';
   if (key == '{')
      return '[';
   if (key == '}')
      return ']';
   if (key == '|')
      return '\\';
   return key;
}

float EaseIn(float start, float end, float a)
{
   return (end - start) * a * a + start;
}

float EaseOut(float start, float end, float a)
{
   return -(end - start) * a * (a - 2) + start;
}

float Bias(float value, float bias)
{
   assert(bias >= 0 && bias <= 1);
   const float kLog25 = log(25);
   bias = .2f * expf(kLog25 * bias); //pow(25,bias)
   return pow(value, bias);
}

float Pow2(float in)
{
   const float kLog2 = log(2);
   return expf(kLog2 * in);
}

void PrintCallstack()
{
#ifdef JUCE_MAC
   void* callstack[128];
   int frameCount = backtrace(callstack, 128);
   char** frameStrings = backtrace_symbols(callstack, frameCount);

   if (frameStrings != nullptr)
   {
      // Start with frame 1 because frame 0 is PrintBacktrace()
      for (int i = 1; i < frameCount; i++)
      {
         printf("%s\n", frameStrings[i]);
      }
      free(frameStrings);
   }
#endif
}

bool IsInUnitBox(ofVec2f pos)
{
   return pos.x >= 0 && pos.x < 1 && pos.y >= 0 && pos.y < 1;
}

std::string GetUniqueName(std::string name, std::vector<IDrawableModule*> existing)
{
   std::string strippedName = name;
   while (strippedName.length() > 1 && CharacterFunctions::isDigit((char)strippedName[strippedName.length() - 1]))
      strippedName.resize(strippedName.length() - 1);
   int suffix = 1;
   std::string suffixString = name;
   ofStringReplace(suffixString, strippedName, "");
   if (!suffixString.empty())
      suffix = atoi(suffixString.c_str());
   while (true)
   {
      bool isNameUnique = true;
      for (int i = 0; i < existing.size(); ++i)
      {
         if (existing[i]->Name() == name)
         {
            ++suffix;
            name = strippedName + ofToString(suffix);
            isNameUnique = false;
            break;
         }
      }
      if (isNameUnique)
         break;
   }

   return name;
}

std::string GetUniqueName(std::string name, std::vector<std::string> existing)
{
   std::string origName = name;
   int suffix = 1;
   while (true)
   {
      bool isNameUnique = true;
      for (int i = 0; i < existing.size(); ++i)
      {
         if (existing[i] == name)
         {
            ++suffix;
            name = origName + ofToString(suffix);
            isNameUnique = false;
            break;
         }
      }
      if (isNameUnique)
         break;
   }

   return name;
}

float DistSqToLine(ofVec2f point, ofVec2f a, ofVec2f b)
{
   float l2 = a.distanceSquared(b);
   if (l2 == 0.0f)
      return point.distanceSquared(a);

   float t = ((point.x - b.x) * (a.x - b.x) + (point.y - b.y) * (a.y - b.y)) / l2;
   if (t < 0.0f)
      return point.distanceSquared(a);
   if (t > 1.0f)
      return point.distanceSquared(b);
   return point.distanceSquared(ofVec2f(b.x + t * (a.x - b.x), b.y + t * (a.y - b.y)));
}

//Jenkins one-at-a-time hash
uint32_t JenkinsHash(const char* key)
{
   uint32_t hash, i;
   for (hash = i = 0; key[i] != 0; ++i)
   {
      hash += key[i];
      hash += (hash << 10);
      hash ^= (hash >> 6);
   }
   hash += (hash << 3);
   hash ^= (hash >> 11);
   hash += (hash << 15);
   return hash;
}

void LoadStateValidate(bool assertion)
{
   if (!assertion)
      throw LoadStateException();
}

double NextBufferTime(bool includeLookahead)
{
   double time = gTime + gBufferSizeMs;
   if (includeLookahead)
      time += TheTransport->GetEventLookaheadMs();
   return time;
}

bool IsAudioThread()
{
   return std::this_thread::get_id() == ModularSynth::GetAudioThreadID();
}

float GetLeftPanGain(float pan)
{
   return 1 - ofClamp(pan, -1, 1);
}

float GetRightPanGain(float pan)
{
   return ofClamp(pan, -1, 1) + 1;
}

void DrawFallbackText(const char* text, float posX, float posY)
{
   static int simplex[95][112] = {
      0,
      16, /* Ascii 32 */
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      10, /* Ascii 33 */
      5,
      21,
      5,
      7,
      -1,
      -1,
      5,
      2,
      4,
      1,
      5,
      0,
      6,
      1,
      5,
      2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      16, /* Ascii 34 */
      4,
      21,
      4,
      14,
      -1,
      -1,
      12,
      21,
      12,
      14,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      21, /* Ascii 35 */
      11,
      25,
      4,
      -7,
      -1,
      -1,
      17,
      25,
      10,
      -7,
      -1,
      -1,
      4,
      12,
      18,
      12,
      -1,
      -1,
      3,
      6,
      17,
      6,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      26,
      20, /* Ascii 36 */
      8,
      25,
      8,
      -4,
      -1,
      -1,
      12,
      25,
      12,
      -4,
      -1,
      -1,
      17,
      18,
      15,
      20,
      12,
      21,
      8,
      21,
      5,
      20,
      3,
      18,
      3,
      16,
      4,
      14,
      5,
      13,
      7,
      12,
      13,
      10,
      15,
      9,
      16,
      8,
      17,
      6,
      17,
      3,
      15,
      1,
      12,
      0,
      8,
      0,
      5,
      1,
      3,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      31,
      24, /* Ascii 37 */
      21,
      21,
      3,
      0,
      -1,
      -1,
      8,
      21,
      10,
      19,
      10,
      17,
      9,
      15,
      7,
      14,
      5,
      14,
      3,
      16,
      3,
      18,
      4,
      20,
      6,
      21,
      8,
      21,
      10,
      20,
      13,
      19,
      16,
      19,
      19,
      20,
      21,
      21,
      -1,
      -1,
      17,
      7,
      15,
      6,
      14,
      4,
      14,
      2,
      16,
      0,
      18,
      0,
      20,
      1,
      21,
      3,
      21,
      5,
      19,
      7,
      17,
      7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      34,
      26, /* Ascii 38 */
      23,
      12,
      23,
      13,
      22,
      14,
      21,
      14,
      20,
      13,
      19,
      11,
      17,
      6,
      15,
      3,
      13,
      1,
      11,
      0,
      7,
      0,
      5,
      1,
      4,
      2,
      3,
      4,
      3,
      6,
      4,
      8,
      5,
      9,
      12,
      13,
      13,
      14,
      14,
      16,
      14,
      18,
      13,
      20,
      11,
      21,
      9,
      20,
      8,
      18,
      8,
      16,
      9,
      13,
      11,
      10,
      16,
      3,
      18,
      1,
      20,
      0,
      22,
      0,
      23,
      1,
      23,
      2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      7,
      10, /* Ascii 39 */
      5,
      19,
      4,
      20,
      5,
      21,
      6,
      20,
      6,
      18,
      5,
      16,
      4,
      15,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      14, /* Ascii 40 */
      11,
      25,
      9,
      23,
      7,
      20,
      5,
      16,
      4,
      11,
      4,
      7,
      5,
      2,
      7,
      -2,
      9,
      -5,
      11,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      14, /* Ascii 41 */
      3,
      25,
      5,
      23,
      7,
      20,
      9,
      16,
      10,
      11,
      10,
      7,
      9,
      2,
      7,
      -2,
      5,
      -5,
      3,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      16, /* Ascii 42 */
      8,
      21,
      8,
      9,
      -1,
      -1,
      3,
      18,
      13,
      12,
      -1,
      -1,
      13,
      18,
      3,
      12,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      26, /* Ascii 43 */
      13,
      18,
      13,
      0,
      -1,
      -1,
      4,
      9,
      22,
      9,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      10, /* Ascii 44 */
      6,
      1,
      5,
      0,
      4,
      1,
      5,
      2,
      6,
      1,
      6,
      -1,
      5,
      -3,
      4,
      -4,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      26, /* Ascii 45 */
      4,
      9,
      22,
      9,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      10, /* Ascii 46 */
      5,
      2,
      4,
      1,
      5,
      0,
      6,
      1,
      5,
      2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      22, /* Ascii 47 */
      20,
      25,
      2,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      20, /* Ascii 48 */
      9,
      21,
      6,
      20,
      4,
      17,
      3,
      12,
      3,
      9,
      4,
      4,
      6,
      1,
      9,
      0,
      11,
      0,
      14,
      1,
      16,
      4,
      17,
      9,
      17,
      12,
      16,
      17,
      14,
      20,
      11,
      21,
      9,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      4,
      20, /* Ascii 49 */
      6,
      17,
      8,
      18,
      11,
      21,
      11,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      14,
      20, /* Ascii 50 */
      4,
      16,
      4,
      17,
      5,
      19,
      6,
      20,
      8,
      21,
      12,
      21,
      14,
      20,
      15,
      19,
      16,
      17,
      16,
      15,
      15,
      13,
      13,
      10,
      3,
      0,
      17,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      15,
      20, /* Ascii 51 */
      5,
      21,
      16,
      21,
      10,
      13,
      13,
      13,
      15,
      12,
      16,
      11,
      17,
      8,
      17,
      6,
      16,
      3,
      14,
      1,
      11,
      0,
      8,
      0,
      5,
      1,
      4,
      2,
      3,
      4,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      6,
      20, /* Ascii 52 */
      13,
      21,
      3,
      7,
      18,
      7,
      -1,
      -1,
      13,
      21,
      13,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      20, /* Ascii 53 */
      15,
      21,
      5,
      21,
      4,
      12,
      5,
      13,
      8,
      14,
      11,
      14,
      14,
      13,
      16,
      11,
      17,
      8,
      17,
      6,
      16,
      3,
      14,
      1,
      11,
      0,
      8,
      0,
      5,
      1,
      4,
      2,
      3,
      4,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      23,
      20, /* Ascii 54 */
      16,
      18,
      15,
      20,
      12,
      21,
      10,
      21,
      7,
      20,
      5,
      17,
      4,
      12,
      4,
      7,
      5,
      3,
      7,
      1,
      10,
      0,
      11,
      0,
      14,
      1,
      16,
      3,
      17,
      6,
      17,
      7,
      16,
      10,
      14,
      12,
      11,
      13,
      10,
      13,
      7,
      12,
      5,
      10,
      4,
      7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      20, /* Ascii 55 */
      17,
      21,
      7,
      0,
      -1,
      -1,
      3,
      21,
      17,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      29,
      20, /* Ascii 56 */
      8,
      21,
      5,
      20,
      4,
      18,
      4,
      16,
      5,
      14,
      7,
      13,
      11,
      12,
      14,
      11,
      16,
      9,
      17,
      7,
      17,
      4,
      16,
      2,
      15,
      1,
      12,
      0,
      8,
      0,
      5,
      1,
      4,
      2,
      3,
      4,
      3,
      7,
      4,
      9,
      6,
      11,
      9,
      12,
      13,
      13,
      15,
      14,
      16,
      16,
      16,
      18,
      15,
      20,
      12,
      21,
      8,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      23,
      20, /* Ascii 57 */
      16,
      14,
      15,
      11,
      13,
      9,
      10,
      8,
      9,
      8,
      6,
      9,
      4,
      11,
      3,
      14,
      3,
      15,
      4,
      18,
      6,
      20,
      9,
      21,
      10,
      21,
      13,
      20,
      15,
      18,
      16,
      14,
      16,
      9,
      15,
      4,
      13,
      1,
      10,
      0,
      8,
      0,
      5,
      1,
      4,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      10, /* Ascii 58 */
      5,
      14,
      4,
      13,
      5,
      12,
      6,
      13,
      5,
      14,
      -1,
      -1,
      5,
      2,
      4,
      1,
      5,
      0,
      6,
      1,
      5,
      2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      14,
      10, /* Ascii 59 */
      5,
      14,
      4,
      13,
      5,
      12,
      6,
      13,
      5,
      14,
      -1,
      -1,
      6,
      1,
      5,
      0,
      4,
      1,
      5,
      2,
      6,
      1,
      6,
      -1,
      5,
      -3,
      4,
      -4,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      3,
      24, /* Ascii 60 */
      20,
      18,
      4,
      9,
      20,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      26, /* Ascii 61 */
      4,
      12,
      22,
      12,
      -1,
      -1,
      4,
      6,
      22,
      6,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      3,
      24, /* Ascii 62 */
      4,
      18,
      20,
      9,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      20,
      18, /* Ascii 63 */
      3,
      16,
      3,
      17,
      4,
      19,
      5,
      20,
      7,
      21,
      11,
      21,
      13,
      20,
      14,
      19,
      15,
      17,
      15,
      15,
      14,
      13,
      13,
      12,
      9,
      10,
      9,
      7,
      -1,
      -1,
      9,
      2,
      8,
      1,
      9,
      0,
      10,
      1,
      9,
      2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      55,
      27, /* Ascii 64 */
      18,
      13,
      17,
      15,
      15,
      16,
      12,
      16,
      10,
      15,
      9,
      14,
      8,
      11,
      8,
      8,
      9,
      6,
      11,
      5,
      14,
      5,
      16,
      6,
      17,
      8,
      -1,
      -1,
      12,
      16,
      10,
      14,
      9,
      11,
      9,
      8,
      10,
      6,
      11,
      5,
      -1,
      -1,
      18,
      16,
      17,
      8,
      17,
      6,
      19,
      5,
      21,
      5,
      23,
      7,
      24,
      10,
      24,
      12,
      23,
      15,
      22,
      17,
      20,
      19,
      18,
      20,
      15,
      21,
      12,
      21,
      9,
      20,
      7,
      19,
      5,
      17,
      4,
      15,
      3,
      12,
      3,
      9,
      4,
      6,
      5,
      4,
      7,
      2,
      9,
      1,
      12,
      0,
      15,
      0,
      18,
      1,
      20,
      2,
      21,
      3,
      -1,
      -1,
      19,
      16,
      18,
      8,
      18,
      6,
      19,
      5,
      8,
      18, /* Ascii 65 */
      9,
      21,
      1,
      0,
      -1,
      -1,
      9,
      21,
      17,
      0,
      -1,
      -1,
      4,
      7,
      14,
      7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      23,
      21, /* Ascii 66 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      13,
      21,
      16,
      20,
      17,
      19,
      18,
      17,
      18,
      15,
      17,
      13,
      16,
      12,
      13,
      11,
      -1,
      -1,
      4,
      11,
      13,
      11,
      16,
      10,
      17,
      9,
      18,
      7,
      18,
      4,
      17,
      2,
      16,
      1,
      13,
      0,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      18,
      21, /* Ascii 67 */
      18,
      16,
      17,
      18,
      15,
      20,
      13,
      21,
      9,
      21,
      7,
      20,
      5,
      18,
      4,
      16,
      3,
      13,
      3,
      8,
      4,
      5,
      5,
      3,
      7,
      1,
      9,
      0,
      13,
      0,
      15,
      1,
      17,
      3,
      18,
      5,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      15,
      21, /* Ascii 68 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      11,
      21,
      14,
      20,
      16,
      18,
      17,
      16,
      18,
      13,
      18,
      8,
      17,
      5,
      16,
      3,
      14,
      1,
      11,
      0,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      19, /* Ascii 69 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      17,
      21,
      -1,
      -1,
      4,
      11,
      12,
      11,
      -1,
      -1,
      4,
      0,
      17,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      18, /* Ascii 70 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      17,
      21,
      -1,
      -1,
      4,
      11,
      12,
      11,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      22,
      21, /* Ascii 71 */
      18,
      16,
      17,
      18,
      15,
      20,
      13,
      21,
      9,
      21,
      7,
      20,
      5,
      18,
      4,
      16,
      3,
      13,
      3,
      8,
      4,
      5,
      5,
      3,
      7,
      1,
      9,
      0,
      13,
      0,
      15,
      1,
      17,
      3,
      18,
      5,
      18,
      8,
      -1,
      -1,
      13,
      8,
      18,
      8,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      22, /* Ascii 72 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      18,
      21,
      18,
      0,
      -1,
      -1,
      4,
      11,
      18,
      11,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      8, /* Ascii 73 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      16, /* Ascii 74 */
      12,
      21,
      12,
      5,
      11,
      2,
      10,
      1,
      8,
      0,
      6,
      0,
      4,
      1,
      3,
      2,
      2,
      5,
      2,
      7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      21, /* Ascii 75 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      18,
      21,
      4,
      7,
      -1,
      -1,
      9,
      12,
      18,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      17, /* Ascii 76 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      0,
      16,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      24, /* Ascii 77 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      12,
      0,
      -1,
      -1,
      20,
      21,
      12,
      0,
      -1,
      -1,
      20,
      21,
      20,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      22, /* Ascii 78 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      18,
      0,
      -1,
      -1,
      18,
      21,
      18,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      21,
      22, /* Ascii 79 */
      9,
      21,
      7,
      20,
      5,
      18,
      4,
      16,
      3,
      13,
      3,
      8,
      4,
      5,
      5,
      3,
      7,
      1,
      9,
      0,
      13,
      0,
      15,
      1,
      17,
      3,
      18,
      5,
      19,
      8,
      19,
      13,
      18,
      16,
      17,
      18,
      15,
      20,
      13,
      21,
      9,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      13,
      21, /* Ascii 80 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      13,
      21,
      16,
      20,
      17,
      19,
      18,
      17,
      18,
      14,
      17,
      12,
      16,
      11,
      13,
      10,
      4,
      10,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      24,
      22, /* Ascii 81 */
      9,
      21,
      7,
      20,
      5,
      18,
      4,
      16,
      3,
      13,
      3,
      8,
      4,
      5,
      5,
      3,
      7,
      1,
      9,
      0,
      13,
      0,
      15,
      1,
      17,
      3,
      18,
      5,
      19,
      8,
      19,
      13,
      18,
      16,
      17,
      18,
      15,
      20,
      13,
      21,
      9,
      21,
      -1,
      -1,
      12,
      4,
      18,
      -2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      16,
      21, /* Ascii 82 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      21,
      13,
      21,
      16,
      20,
      17,
      19,
      18,
      17,
      18,
      15,
      17,
      13,
      16,
      12,
      13,
      11,
      4,
      11,
      -1,
      -1,
      11,
      11,
      18,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      20,
      20, /* Ascii 83 */
      17,
      18,
      15,
      20,
      12,
      21,
      8,
      21,
      5,
      20,
      3,
      18,
      3,
      16,
      4,
      14,
      5,
      13,
      7,
      12,
      13,
      10,
      15,
      9,
      16,
      8,
      17,
      6,
      17,
      3,
      15,
      1,
      12,
      0,
      8,
      0,
      5,
      1,
      3,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      16, /* Ascii 84 */
      8,
      21,
      8,
      0,
      -1,
      -1,
      1,
      21,
      15,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      22, /* Ascii 85 */
      4,
      21,
      4,
      6,
      5,
      3,
      7,
      1,
      10,
      0,
      12,
      0,
      15,
      1,
      17,
      3,
      18,
      6,
      18,
      21,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      18, /* Ascii 86 */
      1,
      21,
      9,
      0,
      -1,
      -1,
      17,
      21,
      9,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      24, /* Ascii 87 */
      2,
      21,
      7,
      0,
      -1,
      -1,
      12,
      21,
      7,
      0,
      -1,
      -1,
      12,
      21,
      17,
      0,
      -1,
      -1,
      22,
      21,
      17,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      20, /* Ascii 88 */
      3,
      21,
      17,
      0,
      -1,
      -1,
      17,
      21,
      3,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      6,
      18, /* Ascii 89 */
      1,
      21,
      9,
      11,
      9,
      0,
      -1,
      -1,
      17,
      21,
      9,
      11,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      20, /* Ascii 90 */
      17,
      21,
      3,
      0,
      -1,
      -1,
      3,
      21,
      17,
      21,
      -1,
      -1,
      3,
      0,
      17,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      14, /* Ascii 91 */
      4,
      25,
      4,
      -7,
      -1,
      -1,
      5,
      25,
      5,
      -7,
      -1,
      -1,
      4,
      25,
      11,
      25,
      -1,
      -1,
      4,
      -7,
      11,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      14, /* Ascii 92 */
      0,
      21,
      14,
      -3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      14, /* Ascii 93 */
      9,
      25,
      9,
      -7,
      -1,
      -1,
      10,
      25,
      10,
      -7,
      -1,
      -1,
      3,
      25,
      10,
      25,
      -1,
      -1,
      3,
      -7,
      10,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      16, /* Ascii 94 */
      6,
      15,
      8,
      18,
      10,
      15,
      -1,
      -1,
      3,
      12,
      8,
      17,
      13,
      12,
      -1,
      -1,
      8,
      17,
      8,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      16, /* Ascii 95 */
      0,
      -2,
      16,
      -2,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      7,
      10, /* Ascii 96 */
      6,
      21,
      5,
      20,
      4,
      18,
      4,
      16,
      5,
      15,
      6,
      16,
      5,
      17,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 97 */
      15,
      14,
      15,
      0,
      -1,
      -1,
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 98 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      11,
      6,
      13,
      8,
      14,
      11,
      14,
      13,
      13,
      15,
      11,
      16,
      8,
      16,
      6,
      15,
      3,
      13,
      1,
      11,
      0,
      8,
      0,
      6,
      1,
      4,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      14,
      18, /* Ascii 99 */
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 100 */
      15,
      21,
      15,
      0,
      -1,
      -1,
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      18, /* Ascii 101 */
      3,
      8,
      15,
      8,
      15,
      10,
      14,
      12,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      12, /* Ascii 102 */
      10,
      21,
      8,
      21,
      6,
      20,
      5,
      17,
      5,
      0,
      -1,
      -1,
      2,
      14,
      9,
      14,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      22,
      19, /* Ascii 103 */
      15,
      14,
      15,
      -2,
      14,
      -5,
      13,
      -6,
      11,
      -7,
      8,
      -7,
      6,
      -6,
      -1,
      -1,
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      19, /* Ascii 104 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      4,
      10,
      7,
      13,
      9,
      14,
      12,
      14,
      14,
      13,
      15,
      10,
      15,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      8, /* Ascii 105 */
      3,
      21,
      4,
      20,
      5,
      21,
      4,
      22,
      3,
      21,
      -1,
      -1,
      4,
      14,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      10, /* Ascii 106 */
      5,
      21,
      6,
      20,
      7,
      21,
      6,
      22,
      5,
      21,
      -1,
      -1,
      6,
      14,
      6,
      -3,
      5,
      -6,
      3,
      -7,
      1,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      17, /* Ascii 107 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      14,
      14,
      4,
      4,
      -1,
      -1,
      8,
      8,
      15,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      8, /* Ascii 108 */
      4,
      21,
      4,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      18,
      30, /* Ascii 109 */
      4,
      14,
      4,
      0,
      -1,
      -1,
      4,
      10,
      7,
      13,
      9,
      14,
      12,
      14,
      14,
      13,
      15,
      10,
      15,
      0,
      -1,
      -1,
      15,
      10,
      18,
      13,
      20,
      14,
      23,
      14,
      25,
      13,
      26,
      10,
      26,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      19, /* Ascii 110 */
      4,
      14,
      4,
      0,
      -1,
      -1,
      4,
      10,
      7,
      13,
      9,
      14,
      12,
      14,
      14,
      13,
      15,
      10,
      15,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 111 */
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      16,
      6,
      16,
      8,
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 112 */
      4,
      14,
      4,
      -7,
      -1,
      -1,
      4,
      11,
      6,
      13,
      8,
      14,
      11,
      14,
      13,
      13,
      15,
      11,
      16,
      8,
      16,
      6,
      15,
      3,
      13,
      1,
      11,
      0,
      8,
      0,
      6,
      1,
      4,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      19, /* Ascii 113 */
      15,
      14,
      15,
      -7,
      -1,
      -1,
      15,
      11,
      13,
      13,
      11,
      14,
      8,
      14,
      6,
      13,
      4,
      11,
      3,
      8,
      3,
      6,
      4,
      3,
      6,
      1,
      8,
      0,
      11,
      0,
      13,
      1,
      15,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      13, /* Ascii 114 */
      4,
      14,
      4,
      0,
      -1,
      -1,
      4,
      8,
      5,
      11,
      7,
      13,
      9,
      14,
      12,
      14,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      17,
      17, /* Ascii 115 */
      14,
      11,
      13,
      13,
      10,
      14,
      7,
      14,
      4,
      13,
      3,
      11,
      4,
      9,
      6,
      8,
      11,
      7,
      13,
      6,
      14,
      4,
      14,
      3,
      13,
      1,
      10,
      0,
      7,
      0,
      4,
      1,
      3,
      3,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      12, /* Ascii 116 */
      5,
      21,
      5,
      4,
      6,
      1,
      8,
      0,
      10,
      0,
      -1,
      -1,
      2,
      14,
      9,
      14,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      10,
      19, /* Ascii 117 */
      4,
      14,
      4,
      4,
      5,
      1,
      7,
      0,
      10,
      0,
      12,
      1,
      15,
      4,
      -1,
      -1,
      15,
      14,
      15,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      16, /* Ascii 118 */
      2,
      14,
      8,
      0,
      -1,
      -1,
      14,
      14,
      8,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      11,
      22, /* Ascii 119 */
      3,
      14,
      7,
      0,
      -1,
      -1,
      11,
      14,
      7,
      0,
      -1,
      -1,
      11,
      14,
      15,
      0,
      -1,
      -1,
      19,
      14,
      15,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      5,
      17, /* Ascii 120 */
      3,
      14,
      14,
      0,
      -1,
      -1,
      14,
      14,
      3,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      9,
      16, /* Ascii 121 */
      2,
      14,
      8,
      0,
      -1,
      -1,
      14,
      14,
      8,
      0,
      6,
      -4,
      4,
      -6,
      2,
      -7,
      1,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      8,
      17, /* Ascii 122 */
      14,
      14,
      3,
      0,
      -1,
      -1,
      3,
      14,
      14,
      14,
      -1,
      -1,
      3,
      0,
      14,
      0,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      39,
      14, /* Ascii 123 */
      9,
      25,
      7,
      24,
      6,
      23,
      5,
      21,
      5,
      19,
      6,
      17,
      7,
      16,
      8,
      14,
      8,
      12,
      6,
      10,
      -1,
      -1,
      7,
      24,
      6,
      22,
      6,
      20,
      7,
      18,
      8,
      17,
      9,
      15,
      9,
      13,
      8,
      11,
      4,
      9,
      8,
      7,
      9,
      5,
      9,
      3,
      8,
      1,
      7,
      0,
      6,
      -2,
      6,
      -4,
      7,
      -6,
      -1,
      -1,
      6,
      8,
      8,
      6,
      8,
      4,
      7,
      2,
      6,
      1,
      5,
      -1,
      5,
      -3,
      6,
      -5,
      7,
      -6,
      9,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      2,
      8, /* Ascii 124 */
      4,
      25,
      4,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      39,
      14, /* Ascii 125 */
      5,
      25,
      7,
      24,
      8,
      23,
      9,
      21,
      9,
      19,
      8,
      17,
      7,
      16,
      6,
      14,
      6,
      12,
      8,
      10,
      -1,
      -1,
      7,
      24,
      8,
      22,
      8,
      20,
      7,
      18,
      6,
      17,
      5,
      15,
      5,
      13,
      6,
      11,
      10,
      9,
      6,
      7,
      5,
      5,
      5,
      3,
      6,
      1,
      7,
      0,
      8,
      -2,
      8,
      -4,
      7,
      -6,
      -1,
      -1,
      8,
      8,
      6,
      6,
      6,
      4,
      7,
      2,
      8,
      1,
      9,
      -1,
      9,
      -3,
      8,
      -5,
      7,
      -6,
      5,
      -7,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      23,
      24, /* Ascii 126 */
      3,
      6,
      3,
      8,
      4,
      11,
      6,
      12,
      8,
      12,
      10,
      11,
      14,
      8,
      16,
      7,
      18,
      7,
      20,
      8,
      21,
      10,
      -1,
      -1,
      3,
      8,
      4,
      10,
      6,
      11,
      8,
      11,
      10,
      10,
      14,
      7,
      16,
      6,
      18,
      6,
      20,
      7,
      21,
      10,
      21,
      12,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
      -1,
   };

   float scale = .5f;
   ofVec2f pen;
   pen.set(posX, posY);

   for (const auto* c = text; *c; ++c)
   {
      const auto ch = *c;

      if (ch == '\n')
      {
         pen.set(posX, pen.y + scale * 60);
         continue;
      }

      const auto* it = simplex[ch - 32];

      const auto nvtcs = *it++;
      const auto spacing = *it++;

      auto from = ofVec2f(-1, -1);

      for (size_t i = 0; i < nvtcs; ++i)
      {
         const auto x = *it++;
         const auto y = *it++;

         const auto to = ofVec2f(x, y);

         if ((from.x != -1 || from.y != -1) && (to.x != -1 || to.y != -1))
            ofLine(pen.x + from.x * scale, pen.y - from.y * scale, pen.x + to.x * scale, pen.y - to.y * scale);

         from = to;
      }

      pen += ofVec2f(spacing * scale, 0);
   }
}

bool EvaluateExpression(std::string expressionStr, float currentValue, float& output)
{
   exprtk::symbol_table<float> symbolTable;
   exprtk::expression<float> expression;
   symbolTable.add_variable("current_value", currentValue);
   symbolTable.add_constants();
   expression.register_symbol_table(symbolTable);

   juce::String input = expressionStr;
   if (input.startsWith("+="))
      input = input.replace("+=", "current_value+");
   if (input.startsWith("*="))
      input = input.replace("*=", "current_value*");
   if (input.startsWith("/="))
      input = input.replace("/=", "current_value/");
   if (input.startsWith("-="))
      input = input.replace("-=", "current_value-");

   exprtk::parser<float> parser;
   bool expressionValid = parser.compile(input.toStdString(), expression);
   if (expressionValid)
   {
      output = expression.value();
      return true;
   }
   return false;
}

ofLog::~ofLog()
{
   std::string output = ofToString(gTime / 1000, 8) + ": " + mMessage;
   DBG(output);
   if (mSendToBespokeConsole)
      TheSynth->LogEvent(output, kLogEventType_Verbose);
}

#ifdef BESPOKE_DEBUG_ALLOCATIONS
FILE* logAllocationsFile;

namespace
{
   const int maxFilenameLen = 90;
   typedef struct
   {
      uint32 address;
      uint32 size;
      char file[maxFilenameLen];
      uint32 line;
      bool allocated;
   } AllocInfo;

   bool enableTracking = false;

   template <class T>
   class my_allocator
   {
   public:
      typedef size_t size_type;
      typedef ptrdiff_t difference_type;
      typedef T* pointer;
      typedef const T* const_pointer;
      typedef T& reference;
      typedef const T& const_reference;
      typedef T value_type;

      my_allocator() {}
      my_allocator(const my_allocator&) {}

      pointer allocate(size_type n, const void* = 0)
      {
         T* t = (T*)malloc(n * sizeof(T));
         return t;
      }

      void deallocate(void* p, size_type)
      {
         if (p)
            free(p);
      }

      pointer address(reference x) const { return &x; }
      const_pointer address(const_reference x) const { return &x; }
      my_allocator<T>& operator=(const my_allocator&) { return *this; }
#undef new
      void construct(pointer p, const T& val)
      {
         new ((T*)p) T(val);
      }
#define new DEBUG_NEW
      void destroy(pointer p)
      {
         p->~T();
      }

      size_type max_size() const { return size_t(-1); }

      template <class U>
      struct rebind
      {
         typedef my_allocator<U> other;
      };

      template <class U>
      my_allocator(const my_allocator<U>&) {}

      template <class U>
      my_allocator& operator=(const my_allocator<U>&) { return *this; }
   };

   map<uint32, AllocInfo*, less<int>, my_allocator<pair<uint32, AllocInfo*> > > allocList;

   void AddTrack(uint32 addr, uint32 asize, const char* fname, uint32 lnum)
   {
      if (enableTracking)
      {
         fprintf(logAllocationsFile, "%-90s:  LINE %5d,  ADDRESS %08x  %8d newed\n", fname, lnum, addr, asize);
         /*AllocInfo* info = (AllocInfo*)malloc(sizeof(AllocInfo));
         
         info->address = addr;
         strncpy(info->file, fname, maxFilenameLen-1);
         info->file[maxFilenameLen-1] = 0;
         info->line = lnum;
         info->size = asize;
         info->allocated = true;
         allocList[addr] = info;*/
      }
   };

   void RemoveTrack(uint32 addr)
   {
      if (enableTracking)
      {
         //fprintf(logAllocationsFile, "ADDRESS %08x  deleted\n", addr);
         /*auto info = allocList.find(addr);
         if (info == allocList.end())
         {
            //assert(false); //Trying to free a never allocated address!
            ofLog() << "Trying to free a never allocated address!";
         }
         else
         {
            if (info->second->allocated == false)
               assert(false); //Trying to free an already freed address!
            else
               info->second->allocated = false;
         }*/
      }
   };
}

void SetMemoryTrackingEnabled(bool enabled)
{
   if (enabled && logAllocationsFile == nullptr)
   {
      logAllocationsFile = fopen(ofToDataPath("allocationslog.txt").c_str(), "w");
   }

   enableTracking = enabled;
}

void DumpUnfreedMemory()
{
   uint32 totalSize = 0;
   char buf[1024];
   map<string, int> perFileTotal;

   for (auto element : allocList)
   {
      const AllocInfo* info = element.second;
      if (info && info->allocated)
      {
         snprintf(buf, sizeof(buf), "%-90s:  LINE %5d,  ADDRESS %08x  %8d unfreed",
                  info->file, info->line, info->address, info->size);
         ofLog() << buf;
         totalSize += info->size;

         perFileTotal[info->file] += info->size;
      }
   }
   ofLog() << "-----------------------------------------------------------";
   for (auto fileInfo : perFileTotal)
   {
      snprintf(buf, sizeof(buf), "%-90s:  %10d unfreed",
               fileInfo.first.c_str(), fileInfo.second);
      ofLog() << buf;
   }
   ofLog() << "-----------------------------------------------------------";
   snprintf(buf, sizeof(buf), "Total Unfreed: %d bytes", totalSize);
   ofLog() << buf;
};

#undef new
void* operator new(std::size_t size) throw(std::bad_alloc)
{
   void* ptr = (void*)malloc(size);
   //AddTrack((uint32)ptr, size, "<unknown>", 0);
   return (ptr);
}
void* operator new(std::size_t size, const char* file, int line) throw(std::bad_alloc)
{
   void* ptr = (void*)malloc(size);
   AddTrack((uint32)ptr, size, file, line);
   return (ptr);
}
void operator delete(void* p) throw()
{
   //RemoveTrack((uint32)p);
   free(p);
}
void* operator new[](std::size_t size) throw(std::bad_alloc)
{
   void* ptr = (void*)malloc(size);
   //AddTrack((uint32)ptr, size, "<unknown>", 0);
   return (ptr);
}
void* operator new[](std::size_t size, const char* file, int line) throw(std::bad_alloc)
{
   void* ptr = (void*)malloc(size);
   AddTrack((uint32)ptr, size, file, line);
   return (ptr);
}
void operator delete[](void* p) throw()
{
   //RemoveTrack((uint32)p);
   free(p);
}
#define new DEBUG_NEW
#else
void SetMemoryTrackingEnabled(bool enabled)
{
}
void DumpUnfreedMemory()
{
   ofLog() << "This only works with BESPOKE_DEBUG_ALLOCATIONS defined";
};
#endif
