// Syd: rock-solid application kernel
// src/kernel/net/socket.rs: socket(2) handler
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

#[expect(deprecated)]
use libc::SOCK_PACKET;
use libc::{
    c_int, AF_ALG, AF_INET, AF_INET6, AF_NETLINK, AF_PACKET, AF_UNIX, SOCK_CLOEXEC, SOCK_NONBLOCK,
    SOCK_RAW,
};
use libseccomp::ScmpNotifResp;
use nix::{
    errno::Errno,
    sys::socket::{AddressFamily, SockFlag, SockType},
};

use crate::{
    compat::{fstatx, STATX_INO},
    cookie::safe_socket,
    debug,
    hook::UNotifyEventRequest,
    log_enabled,
    sandbox::{Flags, NetlinkFamily},
    syslog::LogLevel,
};

#[expect(clippy::cognitive_complexity)]
pub(crate) fn handle_socket(
    request: &UNotifyEventRequest,
    args: &[u64; 6],
    flags: Flags,
    netlink_families: NetlinkFamily,
) -> Result<ScmpNotifResp, Errno> {
    let allow_unsafe_socket = flags.allow_unsafe_socket();
    let allow_unsupp_socket = flags.allow_unsupp_socket();
    let force_cloexec = flags.force_cloexec();
    let force_rand_fd = flags.force_rand_fd();
    let allow_safe_kcapi = flags.allow_safe_kcapi();

    let domain = c_int::try_from(args[0]).or(Err(Errno::EINVAL))?;
    let stype = c_int::try_from(args[1]).or(Err(Errno::EINVAL))?;
    let proto = c_int::try_from(args[2]).or(Err(Errno::EINVAL))?;

    // SAFETY:
    // 1. Limit available domains based on sandbox flags.
    // 2. Deny access to raw & packet sockets,
    //    unless trace/allow_unsafe_socket:1 is set.
    //    Both types require CAP_NET_RAW and use of
    //    SOCK_PACKET is strongly discouraged.
    #[expect(deprecated, reason = "SOCK_PACKET is deprecated")]
    if !allow_unsupp_socket {
        match domain {
            AF_UNIX | AF_INET | AF_INET6 => {}
            AF_ALG if allow_safe_kcapi => {}
            AF_NETLINK => {
                // Restrict AF_NETLINK to the allowlisted families.
                let nlfam = u32::try_from(args[2]).or(Err(Errno::EINVAL))?;
                #[expect(clippy::cast_sign_loss)]
                if nlfam > NetlinkFamily::max() as u32 {
                    return Err(Errno::EINVAL);
                }
                let nlfam = NetlinkFamily::from_bits(1 << nlfam).ok_or(Errno::EINVAL)?;
                if !netlink_families.contains(nlfam) {
                    // SAFETY: Unsafe netlink family, deny.
                    return Err(Errno::EAFNOSUPPORT);
                }
            }
            AF_PACKET if !allow_unsafe_socket => return Err(Errno::EACCES),
            AF_PACKET => {}
            _ => return Err(Errno::EAFNOSUPPORT),
        }
    } else if !allow_safe_kcapi && domain == AF_ALG {
        return Err(Errno::EAFNOSUPPORT);
    } else if !allow_unsafe_socket
        && (domain == AF_PACKET
            || matches!(stype & (SOCK_RAW | SOCK_PACKET), SOCK_RAW | SOCK_PACKET))
    {
        return Err(Errno::EACCES);
    } else {
        // SAFETY: allow_unsupp_socket:1
        // Safe domain, allow.
    }

    let cloexec = force_cloexec || (stype & SOCK_CLOEXEC != 0);
    let stype = stype | SOCK_CLOEXEC;
    let fd = safe_socket(domain, stype, proto)?;

    if log_enabled!(LogLevel::Debug) {
        let inode = fstatx(&fd, STATX_INO)
            .map(|statx| statx.stx_ino)
            .unwrap_or(0);
        let domain = AddressFamily::from_i32(domain)
            .map(|af| format!("{af:?}"))
            .unwrap_or_else(|| "?".to_string());
        let flags = SockFlag::from_bits_retain(stype & (SOCK_CLOEXEC | SOCK_NONBLOCK));
        let stype = SockType::try_from(stype & !(SOCK_CLOEXEC | SOCK_NONBLOCK))
            .map(|st| format!("{st:?}"))
            .unwrap_or_else(|_| "?".to_string());
        debug!("ctx": "net", "op": "create_socket",
                "msg": format!("created {domain} {stype} socket with inode:{inode:#x}"),
                "domain": domain,
                "type": stype,
                "protocol": proto,
                "flags": flags.bits(),
                "inode": inode);
    }

    request.send_fd(fd, cloexec, force_rand_fd)
}
