use crate::app::App;
use crate::infrastructure::TokioRuntime;
use feedly_api::models::SearchResultItem;
use gdk4::Texture;
use glib::{Object, Properties, clone, prelude::*, subclass::*};
use gtk4::{
    Accessible, Actionable, Buildable, CompositeTemplate, ConstraintTarget, ListBoxRow, Widget, prelude::*,
    subclass::prelude::*,
};
use once_cell::sync::Lazy;
use std::cell::RefCell;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::SearchItemRow)]
    #[template(file = "data/resources/ui_templates/discover/search_item.blp")]
    pub struct SearchItemRow {
        #[property(get, set)]
        pub title: RefCell<String>,

        #[property(get, set)]
        pub description: RefCell<String>,

        #[property(get, set, name = "feed-url")]
        pub feed_url: RefCell<String>,

        #[property(get, set, nullable)]
        pub texture: RefCell<Option<Texture>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SearchItemRow {
        const NAME: &'static str = "SearchItemRow";
        type ParentType = ListBoxRow;
        type Type = super::SearchItemRow;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SearchItemRow {
        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> =
                Lazy::new(|| vec![Signal::builder("add-feed").param_types([String::static_type()]).build()]);
            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for SearchItemRow {}

    impl ListBoxRowImpl for SearchItemRow {}

    #[gtk4::template_callbacks]
    impl SearchItemRow {
        #[template_callback]
        fn on_click_released(&self, times: i32, _x: f64, _y: f64) {
            if times != 1 {
                return;
            }

            let row = self.obj();
            row.grab_focus();
            row.emit_by_name::<()>("add-feed", &[&self.feed_url.borrow().clone()]);
        }
    }
}

glib::wrapper! {
    pub struct SearchItemRow(ObjectSubclass<imp::SearchItemRow>)
        @extends Widget, ListBoxRow,
        @implements Accessible, Actionable, Buildable, ConstraintTarget;
}

impl From<SearchResultItem> for SearchItemRow {
    fn from(item: SearchResultItem) -> Self {
        let description = if let Some(description) = &item.description {
            description.replace(['\n', '\r', '_'], " ")
        } else {
            "No description".to_owned()
        };

        let icon_url = if let Some(visual_url) = item.visual_url {
            Some(visual_url)
        } else if let Some(logo) = item.logo {
            Some(logo)
        } else {
            item.icon_url
        };

        let feed_url = item.feed_id.chars().skip(5).collect::<String>();

        let row: SearchItemRow = Object::builder()
            .property("feed-url", feed_url)
            .property("title", item.title.clone().unwrap_or_default())
            .property("description", description)
            .build();

        if let Some(icon_url) = icon_url {
            TokioRuntime::execute_with_callback(
                move || async move {
                    let bytes = App::client().get(&icon_url).send().await?.bytes().await?;
                    Ok(bytes)
                },
                clone!(
                    #[weak]
                    row,
                    move |res: Result<_, reqwest::Error>| {
                        if let Ok(byte_vec) = res {
                            let bytes = glib::Bytes::from_owned(byte_vec);
                            let texture = gdk4::Texture::from_bytes(&bytes);
                            row.set_texture(texture.ok().as_ref());
                        }
                    },
                ),
            );
        }

        row
    }
}
