use crate::app::App;
use crate::gobject_models::GFeed;
use crate::i18n::i18n;
use crate::infrastructure::TokioRuntime;
use glib::{Properties, clone, prelude::*, subclass::*};
use gtk4::{
    Accessible, Box, Buildable, CheckButton, CompositeTemplate, ConstraintTarget, Widget, prelude::*,
    subclass::prelude::*,
};
use libadwaita::{ActionRow, PreferencesGroup, prelude::*};
use news_flash::models::{Feed, FeedID, Url};
use news_flash::{NewsFlash, ParsedUrl};
use once_cell::sync::Lazy;
use std::cell::{Cell, RefCell};
use std::sync::Arc;
use tokio::sync::Semaphore;

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::SelectFeedWidget)]
    #[template(file = "data/resources/ui_templates/add_dialog/select_feed.blp")]
    pub struct SelectFeedWidget {
        #[template_child]
        pub feed_list: TemplateChild<PreferencesGroup>,

        #[property(get, set, nullable, name = "selected-url")]
        pub selected_url: RefCell<Option<String>>,

        #[property(get, set, name = "is-busy")]
        pub is_busy: Cell<bool>,

        #[property(get, set, name = "can-select")]
        pub can_select: Cell<bool>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for SelectFeedWidget {
        const NAME: &'static str = "SelectFeedWidget";
        type ParentType = Box;
        type Type = super::SelectFeedWidget;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for SelectFeedWidget {
        fn constructed(&self) {
            let obj = self.obj();

            obj.connect_selected_url_notify(super::SelectFeedWidget::update_can_select);
            obj.connect_is_busy_notify(super::SelectFeedWidget::update_can_select);
        }

        fn signals() -> &'static [Signal] {
            static SIGNALS: Lazy<Vec<Signal>> = Lazy::new(|| {
                vec![
                    Signal::builder("selected").param_types([GFeed::static_type()]).build(),
                    Signal::builder("error").param_types([String::static_type()]).build(),
                ]
            });
            SIGNALS.as_ref()
        }
    }

    impl WidgetImpl for SelectFeedWidget {}

    impl BoxImpl for SelectFeedWidget {}

    #[gtk4::template_callbacks]
    impl SelectFeedWidget {
        #[template_callback]
        fn visible_button_child(&self, is_busy: bool) -> &'static str {
            if is_busy { "spinner" } else { "text" }
        }

        #[template_callback]
        fn on_select_clicked(&self) {
            let Some(feed_url) = self.selected_url.borrow().clone() else {
                return;
            };
            self.obj().set_is_busy(true);

            let feed_id = FeedID::new(feed_url.as_str());

            TokioRuntime::execute_with_callback(
                || async move {
                    let news_flash = App::news_flash();
                    let semaphore = news_flash
                        .read()
                        .await
                        .as_ref()
                        .map(NewsFlash::get_semaphore)
                        .unwrap_or(Arc::new(Semaphore::new(1)));

                    news_flash::feed_parser::download_and_parse_feed(
                        &Url::parse(&feed_url).unwrap(),
                        &feed_id,
                        None,
                        semaphore,
                        &App::client(),
                    )
                    .await
                },
                clone!(
                    #[weak(rename_to = imp)]
                    self,
                    #[upgrade_or_panic]
                    move |res| {
                        if let Ok(ParsedUrl::SingleFeed(feed)) = res {
                            imp.obj().emit_by_name::<()>("selected", &[&GFeed::from(*feed)]);
                        } else {
                            imp.obj().emit_by_name::<()>("error", &[&i18n("Can't parse Feed")]);
                        }

                        imp.obj().set_is_busy(false);
                    }
                ),
            );
        }
    }
}

glib::wrapper! {
    pub struct SelectFeedWidget(ObjectSubclass<imp::SelectFeedWidget>)
        @extends Widget, Box,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for SelectFeedWidget {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl SelectFeedWidget {
    pub fn fill(&self, feed_vec: Vec<Feed>) {
        let imp = self.imp();

        let mut group: Option<CheckButton> = None;
        for feed in feed_vec {
            let radio_button = CheckButton::new();
            radio_button.set_group(group.as_ref());

            radio_button.connect_toggled(clone!(
                #[strong(rename_to = feed_url)]
                feed.feed_url,
                #[weak(rename_to = this)]
                self,
                move |radio_button| {
                    if radio_button.is_active() {
                        this.set_selected_url(feed_url.clone().map(|url| url.to_string()));
                    }
                }
            ));

            let row = ActionRow::new();
            row.set_title(&feed.label);
            row.set_activatable(true);
            row.set_activatable_widget(Some(&radio_button));
            row.add_prefix(&radio_button);
            row.set_use_markup(false);
            if let Some(url) = feed.feed_url {
                row.set_widget_name(url.as_str());
            }

            imp.feed_list.add(&row);

            group = Some(radio_button);
        }
    }

    fn update_can_select(&self) {
        let is_idle = !self.is_busy();
        let have_selected_url = self.selected_url().is_some();

        self.set_can_select(have_selected_url && is_idle);
    }
}
